package org.nhindirect.gateway.smtp.james.mailet;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;

import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;



import javax.mail.MessagingException;

import javax.mail.internet.MimeMessage;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.mailet.Mail;
import org.apache.mailet.MailAddress;
import org.nhindirect.common.tx.model.Tx;
import org.nhindirect.common.tx.model.TxDetail;
import org.nhindirect.common.tx.model.TxMessageType;
import org.nhindirect.gateway.smtp.MessageProcessResult;
import org.nhindirect.gateway.smtp.james.mailet.SendErrors.MailetPropertiesException;
import org.nhindirect.stagent.NHINDAddress;
import org.nhindirect.stagent.NHINDAddressCollection;

public class NHINDSecurityAndTrustMailetCust extends NHINDSecurityAndTrustMailet {

    /**
     * Overridable method for custom processing before the message is submitted
     * to the SMTP agent.
     *
     * @param mail The incoming mail message.
     */
    boolean debuging = true;

    @Override
    public void init() throws MessagingException {
        try {
            SendErrors.startThread();
        } catch (MailetPropertiesException e) {
            LOGGER.error("Failed to load properties");
            throw new MessagingException("Failed to start SendErrors Thread");
        }
        System.setProperty("jsse.enableCBCProtection", "false");
        super.init();
    }
    private static final Log LOGGER = LogFactory.getFactory().getInstance(
            NHINDSecurityAndTrustMailet.class);

    @Override
    public void service(Mail mail) throws MessagingException {
        Map<String, String> properties = MailetProperties.getPropertiesList();
        if (properties.containsKey("mailet.system") && properties.get("mailet.system").equals("seperate")) {
            onPreprocessMessage(mail);
            onPostprocessMessage(mail, null);
        } else {
            super.service(mail);
        }
    }

    @Override
    protected void onPreprocessMessage(Mail mail) {
        // Log Incoming/Outgoing Message in our database
        if (mail.getSender() != null) {
            // check if message is inbound or outbound
            boolean isInbound = false;
            if (!this.getMailetContext().isLocalServer(
                    mail.getSender().getDomain()
                    .toLowerCase(Locale.getDefault()))) {
                isInbound = true;
            }
            // Get system properties
            Properties properties = mapToProperties(MailetProperties.getPropertiesList());

            // Database connection information
            String db_hostname = properties.getProperty("mailet.db.hostname");
            String db_port = properties.getProperty("mailet.db.port");
            String db_apiname = properties.getProperty("mailet.db.apiname");
            String db_webname = properties.getProperty("mailet.db.webname");
            String db_instance = properties.getProperty("mailet.db.instance");
            String db_apiuserid = properties
                    .getProperty("mailet.db.apiusername");
            String db_webuserid = properties
                    .getProperty("mailet.db.webusername");
            String db_apipassword = properties
                    .getProperty("mailet.db.apipassword");
            String db_webpassword = properties
                    .getProperty("mailet.db.webpassword");

            // Retrieve support email address
            String notificationSender = properties
                    .getProperty("mailet.notification.sender");
            if (!mail
                    .getSender()
                    .toString()
                    .toLowerCase(Locale.getDefault())
                    .equals(notificationSender.toLowerCase(Locale.getDefault()))) {
                // DB Connection setup
                Connection apiconn = null;
                Connection webconn = null;

                // Connect to DB

                try {
                    apiconn = databaseConnect(db_hostname, db_port, db_apiname,
                            db_instance, db_apiuserid, db_apipassword);
                    webconn = databaseConnect(db_hostname, db_port, db_webname,
                            db_instance, db_webuserid, db_webpassword);

                    int log_inboundOutbound = 1;
                    if (!isInbound) {
                        log_inboundOutbound = 0;
                    }

                    // check to see if Message Disposition Notification
                    // will probably always return that it is SMIME rather than
                    // MDN, since MDNs should be encrypted
                    // we need to check again onPostProcess and update the MDN
                    // field in the log
                    int mdn = 0;
                    try {
                        MimeMessage msg = mail.getMessage();
                        final NHINDAddressCollection arecipients = getMailRecipients(mail);
                        final NHINDAddress asender = getMailSender(mail);
                        final Tx txToTrack = getTxToTrack(msg, asender,
                                arecipients);
                        if (txToTrack != null) {
                            if (txToTrack.getMsgType()
                                    .equals(TxMessageType.MDN)) {
                                mdn = 1;
                            }
                        }
                    } catch (MessagingException e1) {
                        LOGGER.debug("Failed to determine if message was an MDN.");
                    }

                    // assume message fails until we update onPostProcess
                    int web_id = LogMessage.logMessageToLog(mail, webconn, 0,
                            log_inboundOutbound, -1); // log message to webmail
                    int api_id = LogMessage.logMessageToLog(mail, apiconn, 0,
                            log_inboundOutbound, mdn); // log message to API

                    mail.setAttribute("log_id_web", web_id);
                    mail.setAttribute("log_id_api", api_id);
                } catch (SQLException sqlException) {
                    LOGGER.debug("SQL Server database not reachable.");
                } finally {
                    if (apiconn != null) {
                        try {
                            apiconn.close();
                        } catch (SQLException closeAPIEx) {
                            LOGGER.debug("Failed to close SQL Database connection.");
                        }
                    }
                    if (webconn != null) {
                        try {
                            webconn.close();
                        } catch (SQLException closeWebEx) {
                            LOGGER.debug("Failed to close SQL Database connection.");
                        }
                    }
                }
            }
        }
    }

    /**
     * Overridable method for custom processing after the message has been
     * processed by the SMTP agent.
     *
     * @param mail The incoming mail message. The contents of the message may
     * have changed from when it was originally received.
     * @param result Contains results of the message processing including the
     * resulting message.
     */
    @SuppressWarnings("CallToThreadDumpStack")
    @Override
    protected void onPostprocessMessage(Mail mail, MessageProcessResult result) {
        // Local Variables
        // Retrieve processed mail information
        Collection<MailAddress> recipients = mail.getRecipients();

        // Get system properties
        Map<String, String> properties = MailetProperties.getPropertiesList();

        // Retrieve support email address
        String notificationSender = properties
                .get("mailet.notification.sender");

        // Database connection information
        String db_hostname = properties.get("mailet.db.hostname");
        String db_port = properties.get("mailet.db.port");
        String db_apiname = properties.get("mailet.db.apiname");
        String db_webname = properties.get("mailet.db.webname");
        String db_instance = properties.get("mailet.db.instance");
        String db_apiuserid = properties.get("mailet.db.apiusername");
        String db_webuserid = properties.get("mailet.db.webusername");
        String db_apipassword = properties.get("mailet.db.apipassword");
        String db_webpassword = properties.get("mailet.db.webpassword");

        // Set up getting message to examine for attachments
        MimeMessage message = null;
        try {
            message = mail.getMessage();
        } catch (MessagingException me) {
            LOGGER.debug("Messaging exception encountered while post-processing message.");
        }
        String log_attachmentTypes;

        // DB Connection setup
        Connection connapi = null;
        Connection connweb = null;

        // Connect to DB
        try {
            connapi = databaseConnect(db_hostname, db_port, db_apiname,
                    db_instance, db_apiuserid, db_apipassword);
            connweb = databaseConnect(db_hostname, db_port, db_webname,
                    db_instance, db_webuserid, db_webpassword);

            if (mail.getSender() != null
                    && !this.getMailetContext().isLocalServer(
                    mail.getSender().getDomain()
                    .toLowerCase(Locale.getDefault()))) {
                if (!mail
                        .getSender()
                        .toString()
                        .toLowerCase(Locale.getDefault())
                        .equals(notificationSender.toLowerCase(Locale
                        .getDefault()))) {
                    // Log Incoming Message in our database
                    // update status in database
                    try {
                        updateLogSuccess(connweb,
                                mail.getAttribute("log_id_web").toString(), 1);
                        updateLogSuccess(connapi,
                                mail.getAttribute("log_id_api").toString(), 1);
                    } catch (SQLException se) {
                        LOGGER.debug("Failed to update status of incoming message during post-processing.");
                    }

                    // attempt to parse attachments
                    log_attachmentTypes = LogMessage.parseAttachments(message);
                    if (log_attachmentTypes != null) { // parse out attachments
                        // if they exist
                        // Query for to update attachments for this message if
                        // necessary
                        try {
                            updateLogAttachments(connweb,
                                    mail.getAttribute("log_id_web").toString(),
                                    log_attachmentTypes);
                            updateLogAttachments(connapi,
                                    mail.getAttribute("log_id_api").toString(),
                                    log_attachmentTypes);
                            LOGGER.info("Updated incoming message log after successful processing.");
                        } catch (SQLException se) {
                            LOGGER.debug("Failed to update attachment types of message during post-processing.");
                        }

                    }
                }
            } else {
                if (mail.getSender() != null
                        && !mail.getSender()
                        .toString()
                        .toLowerCase(Locale.getDefault())
                        .equals(notificationSender.toLowerCase(Locale
                        .getDefault()))) {
                    LOGGER.debug("Message is local");
                    // update status in database
                    try {
                        updateLogSuccess(connweb,
                                mail.getAttribute("log_id_web").toString(), 1);
                        updateLogSuccess(connapi,
                                mail.getAttribute("log_id_api").toString(), 1);
                    } catch (SQLException se) {
                        LOGGER.debug("Failed to update status of outbound message during post-processing.");
                    }
                } else {
                    LOGGER.debug("Message is local notification, do not process for outbound message log.");
                }
            }

            // check to see if it is an MDN
            boolean MDN = false;
            try {

                final NHINDAddressCollection arecipients = getMailRecipients(mail);
                final NHINDAddress asender = getMailSender(mail);
                final Tx txToTrack = getTxToTrack(message, asender, arecipients);
                if (txToTrack != null && message != null) {
                    if (txToTrack.getMsgType() == TxMessageType.MDN) {
                        MDN = true;
                        if (mail.getSender() != null
                                && !this.getMailetContext().isLocalServer(mail.getSender().getDomain().toLowerCase(Locale.getDefault()))) {
                            TxDetail message_id = txToTrack.getDetail("PARENT_MESSAGE_ID");     
                            TxDetail dispo = txToTrack.getDetail("DISPOSTION");
                            if (message_id != null && dispo != null) {
                                message.addHeader("mdn-message", dispo.getDetailValue());
                                long time;
                                 if (mail.getLastUpdated() != null) {
                                   time = mail.getLastUpdated().getTime() / 1000L;
                                 } else {
                                    time = new Date().getTime() / 1000L;
                                }
                                int status_code = 0;
                                if (dispo.getDetailValue().contains("processed")) {
                                   status_code = 1;
                                    LogMessage.update_accounting_disclosure(message_id.getDetailValue(), asender.toString(), time);
                                } else if (dispo.getDetailValue().contains("displayed")) {
                                  status_code = 2;
                                } else if (dispo.getDetailValue().contains("denied")) {
                                    status_code = 3;
                                } else if (dispo.getDetailValue().contains("failed")) {
                                     status_code = 4;
                                } else if (dispo.getDetailValue().contains("error")) {//not standard but in RI code
                                    status_code = 4;
                                } else if (dispo.getDetailValue().contains("dispatched")) {
                                    status_code = 5;
                                }
                                if(status_code > 0) {
                                	LogMessage.setMessageStatus(message_id.getDetailValue(), asender.toString(), status_code, time);
                                }
                          	} else {
                                LOGGER.debug("Could not get data from MDN");
                            }
                        }
                    }
                }
            } catch (MessagingException e1) {
                LOGGER.debug("Failed to see if MDN");
            }
            // if it is an MDN, update MDN status in log
            if (MDN) {
                updateLogMDN(connapi, mail.getAttribute("log_id_api")
                        .toString(), 1);
            }
            // only send external notification if its not an MDN (since MDNs are
            // suppressed from being delivered)
            LOGGER.debug("Recipients list is null: "+(recipients == null));
            if (recipients != null && recipients.size() >= 1 && !MDN) {
            	LOGGER.debug("Recipients list size: "+recipients.size());
                ArrayList<MailAddress> badaddress = LogMessage.sendExternalNotification(connweb, mail);
                LogMessage.proccessMessagesStatus(mail, badaddress);
            }
        } catch (SQLException sqlException) {
            LOGGER.debug("SQL Server database not reachable");
        } finally {
            if (connapi != null) {
                try {
                    connapi.close();
                } catch (SQLException connCloseAPIEx) {
                    LOGGER.debug("SQL Database connection could not be closed.");
                }
            }
            if (connweb != null) {
                try {
                    connweb.close();
                } catch (SQLException connCloseWebEx) {
                    LOGGER.debug("SQL Database connection could not be closed.");
                }
            }
        }
        if (message != null) {
            if (mail.getSender() != null
                    && !this.getMailetContext().isLocalServer(mail.getSender().getDomain().toLowerCase(Locale.getDefault()))) {
                try {
                    message.addHeader("original-id", message.getHeader("message-id", null));
                } catch (MessagingException ex) {
                    LOGGER.debug("could not get message id for storage");
                }
            } else {
                try {
                    message.removeHeader("storage-id");
                } catch (MessagingException ex) {
                    LOGGER.debug("Could not remove storage-id");
                }
                try {
                    message.removeHeader("protected-data");
                } catch (MessagingException ex) {
                    LOGGER.debug("Could not remove protected-data");
                }
            }
            mail.setMessage(message);
        }
    }

    /* private methods */
    public static Properties mapToProperties(Map<String, String> map) {
        Properties p = new Properties();
        p.putAll(map);
        return p;
    }

    /**
     * Establishes a connection with a SQL server database instance and data
     * source, specified by parameters
     */
    private Connection databaseConnect(String db_hostname, String db_port,
            String db_name, String db_instance, String db_userid,
            String db_p) throws SQLException {
        Connection conn = null;
        String db_connect_string = "jdbc:sqlserver://" + db_hostname + ":"
                + db_port + ";databaseName=" + db_name
                + ";ssl=require;hostNameInCertificate=" + db_hostname
                + ";instanceName=" + db_instance + ";portNumber=" + db_port
                + ";trustServerCertificate=true";

        // Load DB Driver
        try {
            Class.forName("com.microsoft.sqlserver.jdbc.SQLServerDriver");
        } catch (ClassNotFoundException f) {
            LOGGER.trace("Could not load SQL Server DB driver");
        }

        // Connect to DB
        try {
            conn = DriverManager.getConnection(db_connect_string, db_userid,
                    db_p);
        } catch (SQLException sqlException) {
            LOGGER.trace("SQL Server database not reachable");
            throw sqlException;
        }

        return conn;
    }

    private void updateLogSuccess(Connection conn, String log_id, int success)
            throws SQLException {
        String updateMessageStatusQuery = "UPDATE mail_log SET success = ? WHERE id = ?";
        PreparedStatement updateMessageLogapi = null;
        try {
            updateMessageLogapi = conn
                    .prepareStatement(updateMessageStatusQuery);
            updateMessageLogapi.setString(1, Integer.toString(success));
            updateMessageLogapi.setString(2, log_id);
            updateMessageLogapi.executeUpdate();
        } catch (SQLException se) {
            LOGGER.debug("SQL Exception while updating status of logged message.");
            throw se;
        } finally {
            if (updateMessageLogapi != null) {
                try {
                    updateMessageLogapi.close();

                } catch (SQLException se) {
                    LOGGER.debug("failed to close statement.");

                }
            }
        }
    }

    private void updateLogMDN(Connection conn, String log_id, int mdn)
            throws SQLException {
        String updateMessageStatusQuery = "UPDATE mail_log SET mdn = ? WHERE id = ?";
        PreparedStatement updateMessageLogapi = null;
        try {
            updateMessageLogapi = conn
                    .prepareStatement(updateMessageStatusQuery);
            updateMessageLogapi.setString(1, Integer.toString(mdn));
            updateMessageLogapi.setString(2, log_id);
            updateMessageLogapi.executeUpdate();
        } catch (SQLException se) {
            LOGGER.debug("SQL Exception while updating MDN status of logged message.");
            throw se;
        } finally {
            if (updateMessageLogapi != null) {
                try {
                    updateMessageLogapi.close();

                } catch (SQLException se) {
                    LOGGER.debug("failed to close statement.");

                }
            }
        }
    }

    private void updateLogAttachments(Connection conn, String log_id,
            String attachments) throws SQLException {
        String updateMessageStatusQuery = "UPDATE mail_log SET attachment_types = ? WHERE id = ?";
        PreparedStatement updateMessageLogapi = null;
        try {
            updateMessageLogapi = conn
                    .prepareStatement(updateMessageStatusQuery);
            updateMessageLogapi.setString(1, attachments);
            updateMessageLogapi.setString(2, log_id);
            updateMessageLogapi.executeUpdate();
        } catch (SQLException se) {
            LOGGER.debug("SQL Exception while updating attachments of logged message.");
            throw se;
        } finally {
            if (updateMessageLogapi != null) {
                try {
                    updateMessageLogapi.close();

                } catch (SQLException se) {
                    LOGGER.debug("failed to close statement.");

                }
            }
        }
    }
}
